<?php

namespace MxoSoftunikPlugin;

use Exception;

/**
 * Class API
 * @package MxoSoftunikPlugin
 */
class API
{
    /** @var string */
    protected $id;

    /** @var string */
    protected $secret;

    /** @var string */
    protected $code;

    /** @var array|object */
    protected $configuration;

    /**
     * API constructor.
     * @param $id
     * @param $secret
     * @param $code
     * @param null $configuration
     */
    public function __construct($id, $secret, $code, $configuration = null)
    {
        $this->id = $id;
        $this->secret = $secret;
        $this->code = $code;

        $this->configuration = $this->returnConfiguration($configuration);
    }

    /**
     * @param null $configuration
     * @return object
     */
    protected function returnConfiguration($configuration = null)
    {
        $resultConfiguration = (object)array(
            'url_entry_point' => 'https://api.glowinthecloud.com',
        );

        if (is_array($configuration)) {
            if (isset($configuration['url_entry_point'])) {
                $resultConfiguration->url_entry_point = $configuration['url_entry_point'];
            }
        } elseif (is_object($configuration)) {
            $resultConfiguration->url_entry_point = $configuration->url_entry_point;
        }

        return $resultConfiguration;
    }

    /**
     * @param $urlEndPoint
     * @param null $getParameters
     * @param null $postParameters
     * @param bool $returnTransfer
     * @param null $headerParameters
     * @return bool|mixed|string
     * @throws Exception
     */
    public function call($urlEndPoint, $getParameters = null, $postParameters = null, $returnTransfer = true, $headerParameters = null)
    {
        $response = null;

        if (!is_array($headerParameters)) {
            $headerParameters = array();
        }

        $curlHandler = curl_init();

        if (!isset($curlHandler)) {
            throw new Exception('Not able to use curl.');
        }

        $getParameterString = '';
        if (is_array($getParameters)) {
            $getParameterString = (!empty($getParameters) ? '?' . http_build_query($getParameters, 'param_', '&', PHP_QUERY_RFC3986) : '');
        }

        $request = "{$this->configuration->url_entry_point}{$urlEndPoint}{$getParameterString}";

        if (!curl_setopt($curlHandler, CURLOPT_URL, $request)) {
            throw new Exception("Not able to set 'CURLOPT_URL'.");
        }

        if (is_array($postParameters) && !empty($postParameters)) {
            $payload = json_encode($postParameters);

            if (!curl_setopt($curlHandler, CURLOPT_POSTFIELDS, $payload)) {
                throw new Exception("Not able to set 'CURLOPT_POSTFIELDS'.");
            }
        }

        $headerParameters['Content-Type'] = 'application/json; charset=utf-8';

        if (!in_array($urlEndPoint, array('/login', '/logout', '/token/validate'))) {
            if (!isset($headerParameters['Authorization'])) {
                throw new Exception("Need an authorization token.");
            }
        }

        $headers = array();
        foreach ($headerParameters as $key => $value) {
            $headers[] = "{$key}: {$value}";
        }

        if (!empty($headers)) {
            if (!curl_setopt($curlHandler, CURLOPT_HTTPHEADER, $headers)) {
                throw new Exception("Not able to set 'CURLOPT_HTTPHEADER'.");
            }
        }

        if ($returnTransfer) {
            if (!curl_setopt($curlHandler, CURLOPT_RETURNTRANSFER, 1)) {
                throw new Exception("Not able to set 'CURLOPT_RETURNTRANSFER'.");
            }
        }

        $result = curl_exec($curlHandler);

        if (false === $result) {
            throw new Exception("Request failed.");
        }

        if ($returnTransfer) {
            $result = json_decode($result, true);
        }

        curl_close($curlHandler);

        return $result;
    }

    /**
     * @param $authorizationToken
     * @return bool
     * @throws Exception
     */
    public function authorizationTokenIsValid($authorizationToken)
    {
        $response = $this->call('/token/validate', array(
            'token' => $authorizationToken,
        ));

        if (isset($response['message']) && $response['message'] === 'OK') {
            return true;
        }

        return false;
    }

    /**
     * @return mixed|null
     * @throws Exception
     */
    public function fetchNewAuthorizationToken()
    {
        $response = $this->call('/login', null, array(
            'id' => $this->id,
            'secret' => $this->secret,
            'code' => $this->code,
        ));

        if (isset($response['message']) && $response['message'] === 'OK' && isset($response['token'])) {
            return $response['token'];
        }

        return null;
    }

    /**
     * @param $authorizationToken
     * @param $parameters
     * @return bool|mixed|string
     * @throws Exception
     */
    public function returnPublicRequestsQueryResult($authorizationToken, $parameters)
    {
        return $this->call('/public-requests', $parameters, null, true, array(
            'Authorization' => $authorizationToken,
        ));
    }

    /**
     * @param $authorizationToken
     * @param $parameters
     * @return bool|mixed|string
     * @throws Exception
     */
    public function returnPublicRequestDetailQueryResult($authorizationToken, $parameters)
    {
        return $this->call('/public-request', $parameters, null, true, array(
            'Authorization' => $authorizationToken,
        ));
    }

    /**
     * @param $authorizationToken
     * @param $parameters
     * @return bool|mixed|string
     * @throws Exception
     */
    public function createCandidate($authorizationToken, $parameters)
    {
        return $this->call('/candidate', null, $parameters, true, array(
            'Authorization' => $authorizationToken,
        ));
    }

    /**
     * @param $authorizationToken
     * @param $parameters
     * @return bool|mixed|string
     * @throws Exception
     */
    public function returnCandidateToken($authorizationToken, $parameters)
    {
        return $this->call('/candidate/check-password', null, $parameters, true, array(
            'Authorization' => $authorizationToken,
        ));
    }

    /**
     * @param $authorizationToken
     * @param $parameters
     * @return bool|mixed|string
     * @throws Exception
     */
    public function createCandidateApplication($authorizationToken, $parameters)
    {
        return $this->call('/public-request/apply', null, $parameters, true, array(
            'Authorization' => $authorizationToken,
        ));
    }
}