<?php

namespace MxoSoftunikPlugin;

use Exception;

/**
 * Class APIHelper
 * @package MxoSoftunikPlugin
 */
class APIHelper
{
    /** @var string */
    const TRANSIENT_AUTHORIZATION_TOKEN_KEY = 'mxo_softunik_api_plugin_authorization_token';

    /** @var API */
    protected $API;

    /** @var mixed */
    protected $authorizationToken;

    /**
     * APIHelper constructor.
     * @param $id
     * @param $secret
     * @param $code
     * @param null $configuration
     */
    public function __construct($id, $secret, $code, $configuration = null)
    {
        $this->API = new API($id, $secret, $code, $configuration);
        $this->authorizationToken = $this->returnCacheAuthorizationToken();
    }

    /**
     * @return mixed
     */
    public function returnCacheAuthorizationToken()
    {
        return get_site_transient(self::TRANSIENT_AUTHORIZATION_TOKEN_KEY);
    }

    /**
     * @throws Exception
     */
    protected function refreshAuthorizationToken(){
        $authorizationToken = $this->API->fetchNewAuthorizationToken();

        if(!isset($authorizationToken)){
            throw new Exception('Not able to get an authorization token.');
        }

        $this->cacheAuthorizationToken($authorizationToken);

        $this->authorizationToken = $authorizationToken;
    }

    /**
     * @param $authorizationToken
     */
    public function cacheAuthorizationToken($authorizationToken)
    {
        set_site_transient(self::TRANSIENT_AUTHORIZATION_TOKEN_KEY,$authorizationToken);
    }

    /**
     * @param $parameters
     * @return bool|mixed|string
     * @throws Exception
     */
    public function returnPublicRequestsQueryResult($parameters)
    {
        $result = $this->API->returnPublicRequestsQueryResult($this->authorizationToken,$parameters);

        if(isset($result['message']) && $result['message'] === 'Invalid token'){
            $this->refreshAuthorizationToken();

            $result = $this->API->returnPublicRequestsQueryResult($this->authorizationToken,$parameters);
        }

        return $result;
    }

    /**
     * @param $parameters
     * @return array|mixed
     * @throws Exception
     */
    public function returnPublicRequestsData($parameters)
    {
        $result = $this->returnPublicRequestsQueryResult($parameters);

        if(isset($result['data'])){
            return $result['data'];
        }

        return array();
    }

    /**
     * @param $parameters
     * @return int|mixed
     * @throws Exception
     */
    public function returnPublicRequestsCount($parameters)
    {
        $result = $this->returnPublicRequestsQueryResult($parameters);

        if(isset($result['count'])){
            return $result['count'];
        }

        return 0;
    }

    /**
     * @param $parameters
     * @return bool|mixed|string
     * @throws Exception
     */
    public function returnPublicRequestDetailQueryResult($parameters)
    {
        $result = $this->API->returnPublicRequestDetailQueryResult($this->authorizationToken,$parameters);

        if(isset($result['message']) && $result['message'] === 'Invalid token'){
            $this->refreshAuthorizationToken();

            $result = $this->API->returnPublicRequestDetailQueryResult($this->authorizationToken,$parameters);
        }

        return $result;
    }

    /**
     * @param $parameters
     * @return array|mixed
     * @throws Exception
     */
    public function returnPublicRequestDetailData($parameters)
    {
        $result = $this->returnPublicRequestDetailQueryResult($parameters);

        if(isset($result['data'])){
            return $result['data'];
        }

        return array();
    }

    /**
     * @param $parameters
     * @return int|mixed
     * @throws Exception
     */
    public function returnPublicRequestDetailCount($parameters)
    {
        $result = $this->returnPublicRequestDetailQueryResult($parameters);

        if(isset($result['count'])){
            return $result['count'];
        }

        return 0;
    }

    /**
     * @param $parameters
     * @return bool|mixed|string
     * @throws Exception
     */
    public function createCandidate($parameters)
    {
        $result = $this->API->createCandidate($this->authorizationToken,$parameters);

        if(isset($result['message']) && $result['message'] === 'Invalid token'){
            $this->refreshAuthorizationToken();

            $result = $this->API->createCandidate($this->authorizationToken,$parameters);
        }

        return $result;
    }

    /**
     * @param $parameters
     * @return bool|mixed|string
     * @throws Exception
     */
    public function returnCandidateToken($parameters)
    {
        $result = $this->API->returnCandidateToken($this->authorizationToken,$parameters);

        if(isset($result['message']) && $result['message'] === 'Invalid token'){
            $this->refreshAuthorizationToken();

            $result = $this->API->returnCandidateToken($this->authorizationToken,$parameters);
        }

        return $result;
    }

    /**
     * @param $parameters
     * @return bool|mixed|string
     * @throws Exception
     */
    public function createCandidateApplication($parameters)
    {
        $result = $this->API->createCandidateApplication($this->authorizationToken,$parameters);

        if(isset($result['message']) && $result['message'] === 'Invalid token'){
            $this->refreshAuthorizationToken();

            $result = $this->API->createCandidateApplication($this->authorizationToken,$parameters);
        }

        return $result;
    }
}