<?php
/**
 * Author: Dev Mxo <dev@agencemxo.com>, Stéphane Cloutier <stephane.cloutier@agencemxo.com>
 * License: MXO Copyright, exclusive usage.
 */

namespace MxoSoftunikPlugin\Front\Site;

use MxoSoftunikPlugin\APIHelper;

/**
 * Class EntryFormSoftunikValidation
 * @package MxoSoftunikPlugin\Front\Site
 */
class EntryFormSoftunikValidation
{
    /** @var EntryFormSoftunikValidation */
    protected static $singleton;

    /** @var APIHelper */
    protected $APIHelper;

    /** @var mixed */
    protected $validationResult;

    /** @var mixed */
    protected $form;

    /** @var mixed */
    protected $entry;

    /** @var EntryFormSoftunikValidationException|EntryFormSoftunikValidationFieldException */
    protected $validationException;

    /**
     * EntryFormSoftunikValidation constructor.
     */
    protected function __construct()
    {
    }

    /**
     * @return EntryFormSoftunikValidation
     */
    public static function singleton()
    {
        if (!isset(self::$singleton)) {
            self::$singleton = new self();
        }

        return self::$singleton;
    }

    /**
     * @param $validationResult
     * @return mixed
     * @throws \Exception
     */
    public static function process($validationResult)
    {
        $singleton = self::singleton();
        return $singleton->_process($validationResult);
    }

    /**
     * @param $validationResult
     * @return mixed
     * @throws \Exception
     */
    protected function _process($validationResult)
    {
        $siteID = get_current_blog_id();

        // Making sure we're only treating the Eastern Canada site for the moment.
        if ($siteID === 3) {
            $softunikConfig = get_option('softunik_config');

            if (!empty($softunikConfig)) {
                $softunikConfig = json_decode($softunikConfig);

                if (!property_exists($softunikConfig, 'locale_mapping')) {
                    throw new \Exception("A valid 'locale_mapping' property is required in Softunik stored config.");
                }

                $locales = array();
                $translationMethodUsed = null;

                // Figuring out if we are using translations and if so the method used for translations.
                if (property_exists($softunikConfig->locale_mapping, 'translations') && !empty($softunikConfig->locale_mapping->translations)) {
                    if (function_exists('pll_languages_list')) {
                        $translationMethodUsed = 'polylang';
                        $polylangSlugs = pll_languages_list(array(
                            'fields' => 'slug',
                        ));
                        $polylangLocales = pll_languages_list(array(
                            'fields' => 'locale',
                        ));
                        foreach ($polylangSlugs as $key => $slug) {
                            $locales[$polylangLocales[$key]] = (object)array(
                                'code2char' => substr($polylangLocales[$key], 0, 2),
                                'slug' => $slug,
                            );
                        }
                    }
                }

                // Feeding a locale to the list if none were added.
                if (empty($locales)) {
                    $locale = get_locale();

                    $locales[$locale] = (object)array(
                        'code2char' => substr($locale, 0, 2),
                    );
                }

                $mainLocale = $softunikConfig->locale_mapping->locale;
                $mainTranslationSlug = (property_exists($locales[$mainLocale], 'slug') ? $locales[$mainLocale]->slug : null);

                $form = $validationResult['form'];
                $entry = \GFFormsModel::get_current_lead();

                if (isset($entry['form_id']) && in_array($entry['form_id'], \MxoMUCarrieresData::returnAllFormIDStrings())) {
                    $this->validationResult = $validationResult;
                    $this->entry = $entry;
                    $this->form = $form;

                    if (isset($this->validationResult['is_valid']) && $this->validationResult['is_valid']) {
                        $error = null;
                        $apply = true;

                        $positionSoftunikId = null;

                        try {
                            $candidateApplicationValues = $this->_return_candidate_application_values();

                            if (!$this->_candidate_application_values_are_valid($candidateApplicationValues)) {
                                $clientMessage = $this->validationException->getClientMessage();
                                throw new EntryFormSoftunikValidationException('Candidate application values are not valid.', 'Invalid values.' . (!empty($clientMessage) ? " {$clientMessage}" : ''));
                            }

                            foreach ($this->form['fields'] as $field) {
                                if (isset($this->entry[$field->id])) {
                                    if ($field->type === 'text' && strpos($field->cssClass, 'sa-position') !== false && !empty($this->entry[$field->id])) {
                                        $positionId = $this->entry[$field->id];

                                        $mainPositionId = $positionId;
                                        if ($translationMethodUsed === 'polylang') {
                                            $pllResult = pll_get_post($positionId, $mainTranslationSlug);
                                            if(!empty($pllResult)){
                                                $mainPositionId = $pllResult;
                                            }
                                        }

                                        if (!empty($mainPositionId)) {
                                            if (function_exists('get_field')) {
                                                $positionSoftunikId = get_field('softunik_public_request_id', $mainPositionId);
                                            } else {
                                                $positionSoftunikId = get_post_meta($mainPositionId, 'softunik_public_request_id', true);
                                            }
                                        }
                                    }
                                }
                            }

                            if (!empty($positionSoftunikId)) {

                                $candidateApplicationValues['position_softunik_id'] = $positionSoftunikId;

                                $candidateValues = $this->_return_candidate_values($candidateApplicationValues);

                                if (empty($candidateValues)) {
                                    $t_candidateValues = $this->_create_candidate_values($candidateApplicationValues);

                                    if (!$this->_candidate_values_are_valid($t_candidateValues)) {
                                        throw new EntryFormSoftunikValidationException('Candidate values are not valid (from create). ' . $this->validationException->getMessage());
                                    }

                                    $softunikCandidateCreationResponse = $this->_create_softunik_candidate($t_candidateValues);

                                    if (!$this->_softunik_candidate_creation_response_is_valid($softunikCandidateCreationResponse)) {
                                        $clientMessage = "";
                                        switch ($softunikCandidateCreationResponse['message']) {
                                            case 'Resource already exists':
                                                $clientMessage = 'Candidate already exists.';
                                                break;
                                        }
                                        throw new EntryFormSoftunikValidationException('Softunik candidate creation response is not valid.' . (isset($softunikCandidateCreationResponse['message']) ? " ({$softunikCandidateCreationResponse['message']})." : ''), $clientMessage);
                                    }

                                    $candidateValues = $t_candidateValues;
                                }

                                if (!$this->_candidate_values_are_valid($candidateValues)) {
                                    throw new EntryFormSoftunikValidationException('Candidate values are not valid. ' . $this->validationException->getMessage());
                                }

                                $softunikCandidateTokenResponse = $this->_return_softunik_candidate_token($candidateValues);

                                if (!$this->_softunik_candidate_token_response_is_valid($softunikCandidateTokenResponse)) {
                                    throw new EntryFormSoftunikValidationException('Softunik candidate token response is not valid.' . (isset($softunikCandidateTokenResponse['message']) ? " ({$softunikCandidateTokenResponse['message']})." : ''));
                                }

                                $candidateToken = $softunikCandidateTokenResponse['data']['token'];

                                if (!empty($candidateToken) && !empty($candidateValues) && !empty($candidateApplicationValues)) {

                                    $softunikCandidateApplicationCreationResponse = $this->_create_softunik_candidate_application($candidateToken, $candidateValues, $candidateApplicationValues);

                                    if (!$this->_softunik_candidate_application_creation_response_is_valid($softunikCandidateApplicationCreationResponse)) {
                                        $clientMessage = "";
                                        switch ($softunikCandidateApplicationCreationResponse['message']) {
                                            case 'Resource already exists':
                                                $clientMessage = 'Application was already made.';
                                                break;
                                        }
                                        throw new EntryFormSoftunikValidationException('Softunik candidate application creation response is not valid.' . (isset($softunikCandidateApplicationCreationResponse['message']) ? " ({$softunikCandidateApplicationCreationResponse['message']})." : ''), $clientMessage);
                                    }
                                }
                            }
                        } catch (EntryFormSoftunikValidationException $e) {
                            $clientMessage = $e->getClientMessage();
                            $error = 'Unable to post your application.' . (!empty($clientMessage) ? " {$clientMessage}" : '');
                            $apply = false;
                        }

                        if (!$apply) {
                            $this->validationResult['is_valid'] = false;
                            if (isset($error)) {
                                add_filter("gform_validation_message_{$entry['form_id']}", function ($message, $form) use ($error) {
                                    return "<div class='validation_error'>" . $error . '</div>';
                                }, 10, 2);
                            }
                        }
                    }

                    $this->validationResult['form'] = $this->form;

                    return $this->validationResult;
                }
            }
        }

        return $validationResult;
    }

    /**
     * @return array
     */
    protected function _return_candidate_application_values()
    {
        $candidateApplicationValues = [];

        foreach ($this->form['fields'] as $field) {
            if (isset($this->entry[$field->id])) {
                if ($field->type === 'text' && strpos($field->cssClass, 'sa-firstname') !== false && !empty($this->entry[$field->id])) {
                    $candidateApplicationValues['firstname_field_id'] = $field->id;
                    $candidateApplicationValues['firstname'] = $this->entry[$field->id];
                } elseif ($field->type === 'text' && strpos($field->cssClass, 'sa-name') !== false && !empty($this->entry[$field->id])) {
                    $candidateApplicationValues['name_field_id'] = $field->id;
                    $candidateApplicationValues['name'] = $this->entry[$field->id];
                } elseif ($field->type === 'email' && strpos($field->cssClass, 'sa-email') !== false && !empty($this->entry[$field->id])) {
                    $candidateApplicationValues['email_field_id'] = $field->id;
                    $candidateApplicationValues['email'] = $this->entry[$field->id];
                } elseif ($field->type === 'text' && strpos($field->cssClass, 'sa-position') !== false && !empty($this->entry[$field->id])) {
                    $candidateApplicationValues['position_field_id'] = $field->id;
                    $candidateApplicationValues['position'] = $this->entry[$field->id];
                } /** @noinspection SpellCheckingInspection */ elseif ($field->type === 'fileupload' && strpos($field->cssClass, 'sa-document') !== false && !empty($this->entry[$field->id])) {
                    $candidateApplicationValues['document_field_id'] = $field->id;
                    $candidateApplicationValues['document'] = $this->entry[$field->id];
                } elseif ($field->type === 'textarea' && strpos($field->cssClass, 'sa-message') !== false && !empty($this->entry[$field->id])) {
                    $candidateApplicationValues['message_field_id'] = $field->id;
                    $candidateApplicationValues['message'] = $this->entry[$field->id];
                }
            }
        }

        return $candidateApplicationValues;
    }

    /**
     * @param $candidateApplicationValues
     * @return bool
     */
    protected function _candidate_application_values_are_valid($candidateApplicationValues)
    {
        try {
            if (!isset($candidateApplicationValues['firstname'])) {
                throw new EntryFormSoftunikValidationFieldException('firstname', 'Firstname must be filled.', 'Firstname must be filled.');
            }

            if (!isset($candidateApplicationValues['name'])) {
                throw new EntryFormSoftunikValidationFieldException('name', 'Name must be filled.', 'Name must be filled.');
            }

            if (!isset($candidateApplicationValues['email'])) {
                throw new EntryFormSoftunikValidationFieldException('email', 'Email must be filled.', 'Email must be filled.');
            }

            if (!isset($candidateApplicationValues['position'])) {
                throw new EntryFormSoftunikValidationFieldException('position', 'A position must be selected.', 'A position must be selected.');
            }

            if (isset($candidateApplicationValues['document'])) {
                $document = $_FILES['input_' . $candidateApplicationValues['document_field_id']];

                if (isset($document['error']) && $document['error'] != 0) {
                    throw new EntryFormSoftunikValidationFieldException('document', "Document could not be uploaded [error: {$document['error']}].", 'Document could not be uploaded.');
                }
            }

            if (!isset($candidateApplicationValues['message'])) {
                throw new EntryFormSoftunikValidationFieldException('message', 'Message must be filled.', 'Message must be filled.');
            }
        } catch (EntryFormSoftunikValidationFieldException $e) {
            $this->validationException = $e;
            return false;
        }

        return true;
    }

    /**
     * @param $values
     * @return array
     */
    protected function _return_candidate_values($values)
    {
        global $wpdb;

        $candidateValues = [];

        $sql = "" .
            "SELECT " .
            "   password " .
            "FROM " . $wpdb->base_prefix . "mxo_candidate " .
            "WHERE " .
            "   email=%s " .
            "";
        $stmt = $wpdb->prepare($sql, $values['email']);
        $result = $wpdb->get_row($stmt);

        if (!empty($result)) {
            $candidateValues['email'] = $values['email'];
            $candidateValues['password'] = $result->password;
            $candidateValues['firstname'] = $values['firstname'];
            $candidateValues['name'] = $values['name'];

            if (isset($values['document'])) {
                $document = $_FILES['input_' . $values['document_field_id']];

                $candidateValues['file'] = [
                    'name' => $document['name'],
                    'type' => $document['type'],
                    'content' => base64_encode(file_get_contents($_FILES['input_' . $values['document_field_id']]['tmp_name'])),
                ];
            }
        }

        return $candidateValues;
    }

    /**
     * @param $values
     * @return array
     */
    protected function _create_candidate_values($values)
    {
        global $wpdb;

        $sql = "" .
            "INSERT " .
            "INTO " . $wpdb->base_prefix . "mxo_candidate " .
            "( " .
            "   email, " .
            "   password " .
            ") " .
            "VALUES ( " .
            "   %s, " .
            "   %s " .
            ") " .
            "";

        $stmt = $wpdb->prepare($sql, $values['email'], md5($values['email']));
        $wpdb->query($stmt);

        return $this->_return_candidate_values($values);
    }

    /**
     * @param $candidateValues
     * @return bool
     */
    protected function _candidate_values_are_valid($candidateValues)
    {
        try {
            if (!isset($candidateValues['email'])) {
                throw new EntryFormSoftunikValidationException('Email is needed.');
            }

            if (!isset($candidateValues['password'])) {
                throw new EntryFormSoftunikValidationException('Password is needed.');
            }
        } catch (EntryFormSoftunikValidationException $e) {
            $this->validationException = $e;
            return false;
        }

        return true;
    }

    /**
     * @param $candidateValues
     * @return bool|mixed|string
     * @throws \Exception
     */
    protected function _create_softunik_candidate($candidateValues)
    {
        $APIHelper = $this->_retrieve_API_helper();

        $parameters = [
            'email' => $candidateValues['email'],
            'password' => $candidateValues['password'],
            'firstname' => $candidateValues['firstname'],
            'name' => $candidateValues['name'],
        ];

        if (isset($candidateValues['file'])) {
            $parameters['file'] = $candidateValues['file'];
        }

        return $APIHelper->createCandidate($parameters);
    }

    /**
     * @param $softunikCandidateCreationResponse
     * @return bool
     */
    protected function _softunik_candidate_creation_response_is_valid($softunikCandidateCreationResponse)
    {
        return (
            isset($softunikCandidateCreationResponse['message'])
            && $softunikCandidateCreationResponse['message'] === 'OK'
        );
    }

    /**
     * @param $candidateValues
     * @return bool|mixed|string
     * @throws \Exception
     */
    protected function _return_softunik_candidate_token($candidateValues)
    {
        $APIHelper = $this->_retrieve_API_helper();

        return $APIHelper->returnCandidateToken(
            array(
                'email' => $candidateValues['email'],
                'password' => $candidateValues['password'],
            )
        );
    }

    /**
     * @param $softunikCandidateTokenResponse
     * @return bool
     */
    protected function _softunik_candidate_token_response_is_valid($softunikCandidateTokenResponse)
    {
        return (
            isset($softunikCandidateTokenResponse['message'])
            && $softunikCandidateTokenResponse['message'] === 'OK'
            && isset($softunikCandidateTokenResponse['data']['token'])
        );
    }

    /**
     * @param $candidateToken
     * @param $candidateValues
     * @param $candidateApplicationValues
     * @return bool|mixed|string
     * @throws \Exception
     */
    protected function _create_softunik_candidate_application($candidateToken, $candidateValues, $candidateApplicationValues)
    {
        $APIHelper = $this->_retrieve_API_helper();

        return $APIHelper->createCandidateApplication(
            array(
                'token' => $candidateToken,
                'id' => $candidateApplicationValues['position_softunik_id'],
            )
        );
    }

    /**
     * @param $softunikCandidateApplicationCreationResponse
     * @return bool
     */
    protected function _softunik_candidate_application_creation_response_is_valid($softunikCandidateApplicationCreationResponse)
    {
        return (
            isset($softunikCandidateApplicationCreationResponse['message'])
            && $softunikCandidateApplicationCreationResponse['message'] === 'OK'
        );
    }

    /**
     * @return APIHelper
     */
    protected function _retrieve_API_helper()
    {
        if (!isset($this->APIHelper)) {
            /** @noinspection SpellCheckingInspection */
            $this->APIHelper = new APIHelper(
                'APP-API-PROGRAIN',
                'AaVMV8BkUdzxcxK',
                '10142'
            );
        }

        return $this->APIHelper;
    }
}
